import React, { useEffect, useState } from 'react';

function App() {
  const [product, setProduct] = useState(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState('');
  const [saving, setSaving] = useState(false);
  const [success, setSuccess] = useState('');
  const [initialLoad, setInitialLoad] = useState(true);

  // These would be set from your config/environment
  const leapHost = '/apps/secure/';
  const leapAppId = '49104c7d-48b1-45a4-85cb-ba684880ca6b';
  const leapFormId = 'F_Product';

  useEffect(() => {
    const handleSelectedProductEvent = (e) => {
      setInitialLoad(false);
      fetchProduct(e.detail.message);
    };

    window.addEventListener('selected-product-event', handleSelectedProductEvent);

    // On first load, just show "select a product" message
    setLoading(false);
    setInitialLoad(true);
    setProduct(null);

    return () => {
      window.removeEventListener('selected-product-event', handleSelectedProductEvent);
    };
    // eslint-disable-next-line
  }, []);

  const fetchProduct = (id) => {
    setLoading(true);
    setError('');
    fetch(
      `${leapHost}org/data/${leapAppId}/${leapFormId}/${id}?format=application/json`
    )
      .then((res) => {
        if (!res.ok) throw new Error('Failed to fetch product');
        return res.json();
      })
      .then((data) => {
        setProduct(data.items[0]);
        setLoading(false);
      })
      .catch((err) => {
        setError(err.message);
        setLoading(false);
      });
  };

  const updateProduct = async (leapItem, submitButton) => {
    setSaving(true);
    setSuccess('');
    setError('');
    try {
      const freedomIdentifyKey = new Date().getTime();
      document.cookie = `freedomIdentifyKey=${freedomIdentifyKey}; path=/`;

      const url = `${leapHost}org/data/${leapAppId}/${leapFormId}/${leapItem.uid}?freedomIdentifyKey=${freedomIdentifyKey}`;
      const payload = {
        pressedButton: submitButton,
        flowState: leapItem.flowState,
        uid: leapItem.uid,
      };
      Object.keys(leapItem)
        .filter((k) => k.startsWith('F_'))
        .forEach((entry) => {
          payload[entry] = leapItem[entry];
        });

      const options = {
        method: 'PUT',
        headers: {
          'Content-Type': 'application/json',
          Accept: 'application/json',
        },
        body: JSON.stringify(payload),
      };

      const response = await fetch(url, options);
      const data = await response.json();

      if (!response.ok) {
        const errorMsg = (data && data.message) || response.statusText;
        throw new Error(errorMsg);
      }

      setSuccess('Product updated successfully!');
      setSaving(false);
      //fetchProduct(leapItem.uid);
      window.dispatchEvent(new CustomEvent('updated-product-event'));
    } catch (err) {
      setError(err.message);
      setSaving(false);
    }
  };

  const handleChange = (e) => {
    const { name, value } = e.target;
    setProduct((prev) => ({
      ...prev,
      [name]: value,
    }));
  };

  const handleSubmit = (e) => {
    e.preventDefault();
    if (!product) return;
    updateProduct(product, 'S_Update');
  };

  if (loading)
    return (
      <div className="p-4 d-flex justify-content-center align-items-center">
        <div className="text-center">
          <div className="spinner-border text-primary" role="status"></div>
          <p className="mt-3">Loading product details...</p>
        </div>
      </div>
    );

  if (initialLoad && !product)
    return (
      <div className="p-4 d-flex justify-content-center align-items-center">
        <div className="alert alert-info text-center">Please select a product to view details.</div>
      </div>
    );

  if (error)
    return (
      <div className="p-4 d-flex justify-content-center align-items-center">
        <div className="alert alert-danger text-center">{error}</div>
      </div>
    );
  if (!product)
    return (
      <div className="p-4 d-flex justify-content-center align-items-center">
        <div className="alert alert-warning text-center">No product found.</div>
      </div>
    );

  return (
    <div className="p-4 d-flex justify-content-center align-items-center">
      <div className="card shadow-3 w-100">
        <div className="card-header bg-primary text-white text-center">
          <h2 className="mb-0">Product Details</h2>
        </div>
        <form className="card-body" onSubmit={handleSubmit}>
          <div className="row">
            <div className="col-md-6">
              <div className="mb-2">
                <label className="form-label" htmlFor="F_Name1">Name</label>
                <input type="text" className="form-control" id="F_Name1" name="F_Name1" value={product.F_Name1} onChange={handleChange} required />
              </div>
              <div className="mb-2">
                <label className="form-label" htmlFor="F_Category1">Category</label>
                <input type="text" className="form-control" id="F_Category1" name="F_Category1" value={product.F_Category1} onChange={handleChange} />
              </div>
              <div className="mb-2">
                <label className="form-label" htmlFor="F_Description1">Description</label>
                <textarea className="form-control" id="F_Description1" name="F_Description1" value={product.F_Description1} onChange={handleChange} rows={3} />
              </div>
              <div className="mb-2">
                <label className="form-label" htmlFor="F_SKU1">SKU</label>
                <input type="text" className="form-control" id="F_SKU1" name="F_SKU1" value={product.F_SKU1} onChange={handleChange} />
              </div>
              <div className="mb-2">
                <label className="form-label" htmlFor="F_Price1">Price</label>
                <input type="text" className="form-control" id="F_Price1" name="F_Price1" value={product.F_Price1} onChange={handleChange} />
              </div>
            </div>
            <div className="col-md-6">
              <div className="mb-2">
                <label className="form-label" htmlFor="F_StockQuantity1">Stock Quantity</label>
                <input type="text" className="form-control" id="F_StockQuantity1" name="F_StockQuantity1" value={product.F_StockQuantity1} onChange={handleChange} />
              </div>
              <div className="mb-2">
                <label className="form-label" htmlFor="F_Supplier1">Supplier</label>
                <input type="text" className="form-control" id="F_Supplier1" name="F_Supplier1" value={product.F_Supplier1} onChange={handleChange} />
              </div>
              <div className="mb-2">
                <label className="form-label" htmlFor="F_Rating1">Rating</label>
                <input type="text" className="form-control" id="F_Rating1" name="F_Rating1" value={product.F_Rating1} onChange={handleChange} />
              </div>
              <div className="mb-2">
                <label className="form-label" htmlFor="F_DateAdded1">Date Added</label>
                <input type="date" className="form-control" id="F_DateAdded1" name="F_DateAdded1" value={product.F_DateAdded1} onChange={handleChange} />
              </div>
            </div>
          </div>
          <div className="text-end mt-3">
            <button type="submit" className="btn btn-primary" disabled={saving}>
              {saving ? (<><span className="spinner-border spinner-border-sm me-2"></span>Saving...</>) : 'Save Changes'}
            </button>
          </div>
          {success && <div className="alert alert-success mt-3 text-center">{success}</div>}
          {error && <div className="alert alert-danger mt-3 text-center">{error}</div>}
        </form>
        <div className="card-footer text-muted text-end">
          Last updated: {new Date(product.lastModified).toLocaleString()}
        </div>
      </div>
    </div>
  );
}

export default App;